/* commands.c -- functions to get commands and control r8c
 * Copyright (C) 2008  theDark
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>

#include "definitions.h"
#include "handlers.h"
#include "commands.h"


int splitstr(const char *in, char *key, unsigned int keylen,
	     char *command, unsigned int commandlen) {
	char buffer[BUF];
	char *ret;

	strncpy(buffer, in, BUF);
	buffer[BUF-1] = '\0';

	if((ret = strtok(buffer, "\n")) == NULL)
		return -1;
	if((ret = strtok(ret, " ")) == NULL)
		return -1;
	strncpy(key, ret, keylen);
	key[keylen-1] = '\0';
	if((ret = strtok(NULL, "")) == NULL)
		return -1;
	strncpy(command, ret, commandlen);
	command[commandlen-1] = '\0';

	return 0;
}

int make_stepper_command(int *xpos, int *ypos, const char *in,
			 char *out, unsigned int len) {
	int i, num, value, pos;
	char direction[8];

	if(*in == 'x') {
		num = 1;
		pos = *xpos;
	}
	else if(*in == 'y') {
		num = 2;
		pos = *ypos;
	}
	else
		return -1;
	
	for(i = 2; i < strlen(in); i++) {
		if(in[i] < '0' || in[i] > '9')
			return -1;
	}

	value = atoi(in + 2);

	if(in[1] == '+')
		pos += value;
	else if(in[1] == '-')
		pos -= value;
	else if(in[1] == '=')
		pos = value;
	else
		return -1;
	
	if(pos < 0 || (*in == 'x' && pos > XPOS_MAX)
	   || (*in == 'y' && pos > YPOS_MAX))
	   	return -1;

	if(pos > *xpos && *in == 'x') {
		strncpy(direction, "rew", 8);
		value = pos - *xpos;
	}
	else if(pos > *ypos && *in == 'y') {
		strncpy(direction, "rew", 8);
		value = pos - *ypos;
	}
	else if(pos < *xpos && *in == 'x') {
		strncpy(direction, "fwd", 8);
		value = *xpos - pos;
	}
	else if(pos < *ypos && *in == 'y') {
		strncpy(direction, "fwd", 8);
		value = *ypos - pos;
	}
	else
		return -1;

	snprintf(out, len, "stepper %d %s %d\n", num, direction, value);

	if(*in == 'x')
		*xpos = pos;
	else
		*ypos = pos;
	
	return 0;
}

int is_analog(const char *in) {
	int i;
	char buffer[8];

	strncpy(buffer, in, 8);
	buffer[strlen(buffer) - 1] = '\0';	// remove '\n'
	
	for(i = 0; i < strlen(buffer); i++) {
		if(buffer[i] < '0' || buffer[i] > '9')
			return 0;
	}

	return 1;
}
	
int r8c_sr(FILE *r, FILE *w, const char *sendstr, char *recvstr, unsigned int len) {
	char buffer[BUF];
	int i, fail = 1;

	for(i = 0; i < MAX_RETRIES + 1; i++) {
		if(fputs(sendstr, w) < 0)
			return -3;
	
		if(fgets(buffer, BUF, r) == NULL)
			return -3;
	
		if(strcmp(sendstr, "analog\n") == 0) {
			if(is_analog(buffer)) {
				fail = 0;
				break;
			}
#if DEBUG
			else {
				fprintf(stderr, "[%d]: WARNING: R8C did not return analog value"
					", error follows...\n", getpid());
				fprintf(stderr, "%s", buffer);
			}
#endif
		}
		else {
			if(strcmp(buffer, "OK\n") == 0) {
				fail = 0;
				break;
			}
#if DEBUG
			else {
				fprintf(stderr, "[%d]: WARNING: R8C did not return OK"
					", error follows...\n", getpid());
				fprintf(stderr, "%s", buffer);
			}
#endif
		}
	}

	if(recvstr != NULL)
		strncpy(recvstr, buffer, len);

	if(fail) {
#if DEBUG
		fprintf(stderr, "[%d]: FATAL: Maximum number of retries"
			" has been exceeded\n", getpid());
#endif
		return -4;
	}

	return 0;
}
		

void make_c_error(const char *fname, int error) {
	char buffer[BUF];

	switch(error) {
	    case -1:
		snprintf(buffer, BUF, "[cxa]: In function %s(): Error on "
			 "reading or writing from or to main", fname);
		perror(buffer);
		break;

	    case -2:
		fprintf(stderr, "[cxa]: In function %s(): main did "
			"not send cancel, but other stuff\n", fname);
		break;
		
	    case -3:
		snprintf(buffer, BUF, "[cxa]: In function %s(): Error on "
			 "reading or writing from or to SERIAL", fname);
		perror(buffer);
		break;

	    case -4:
		fprintf(stderr, "[cxa]: In function %s(): R8C did "
			"not return OK, but other stuff\n", fname);
		break;

	    case -5:
		snprintf(buffer, BUF, "[cxa]: In function %s(): Error on "
			 "opening PLOT_DATFILE", fname);
		perror(buffer);
		break;

	    case -6:
		snprintf(buffer, BUF, "[cxa]: In function %s(): Error on "
			 "opening pipe to gnuplot", fname);
		perror(buffer);
		break;

	    default:
	    	fprintf(stderr, "[cxa]: After function %s(): Unknown "
			"errno: %d\n", fname, error);
	}
	die_at_once();
}

int c_start(int x, int y, int sock, FILE *r, FILE *w ) {
	int ret;
	char buffer[BUF];

	snprintf(buffer, BUF, "OK\n");
	if(send(sock, buffer, strlen(buffer)+1, 0) < 0)
		return -1;


	if(y > 0) {
		snprintf(buffer, BUF, "stepper 2 fwd %d\n", y);
		if((ret = r8c_sr(r, w, buffer, NULL, 0)) < 0)
			return ret;
	}


	if((ret = recv(sock, buffer, BUF, MSG_DONTWAIT)) < 0 && errno != EAGAIN)
		return -1;
	if(ret >= 0) {
		if(strcmp(buffer, "cancel\n") == 0) {
			snprintf(buffer, BUF, "POS %d 0\n", x);
			if(send(sock, buffer, strlen(buffer)+1, 0) < 0)
				return -1;

			snprintf(buffer, BUF, "READY\n");
			if(send(sock, buffer, strlen(buffer)+1, 0) < 0)
				return -1;

			return 0;
		}
		else
			return -2;
	}


	snprintf(buffer, BUF, "POS %d 0\n", x);
	if(send(sock, buffer, strlen(buffer)+1, MSG_DONTWAIT) < 0 && 
	   errno != EAGAIN)
		return -1;


	if(x > 0) {
		snprintf(buffer, BUF, "stepper 1 fwd %d\n", x);
		if((ret = r8c_sr(r, w, buffer, NULL, 0)) < 0)
			return ret;
	}


	snprintf(buffer, BUF, "POS 0 0\n");
	if(send(sock, buffer, strlen(buffer)+1, 0) < 0)
		return -1;
	
	snprintf(buffer, BUF, "READY\n");
	if(send(sock, buffer, strlen(buffer)+1, 0) < 0)
		return -1;

	return 0;
}

int c_xy(int x, int y, int sock, FILE *r, FILE *w, const char *command) {
	int ret;
	char buffer[BUF];
	char sendstr[BUF];

	if(make_stepper_command(&x, &y, command, sendstr, BUF) < 0) {
		snprintf(buffer, BUF, "INVALID_VALUE\n");
		if(send(sock, buffer, strlen(buffer)+1, 0) < 0)
			return -1;

		return 0;
	}

	snprintf(buffer, BUF, "OK\n");
	if(send(sock, buffer, strlen(buffer)+1, 0) < 0)
		return -1;

#if DEBUG > 1
	printf("[cxa]: In function c_xy(): sending: %s", sendstr);
#endif

	if((ret = r8c_sr(r, w, sendstr, NULL, 0)) < 0)
		return ret;


	snprintf(buffer, BUF, "POS %d %d\n", x, y);
	if(send(sock, buffer, strlen(buffer)+1, 0) < 0)
		return -1;
	
	snprintf(buffer, BUF, "READY\n");
	if(send(sock, buffer, strlen(buffer)+1, 0) < 0)
		return -1;

	return 0;
}

int c_scan(int x, int y, int sock, FILE *r, FILE *w, const char *command) {
	int ret, value;
	int i, j, k;
	int xend = 0, yend = 0;
	char buffer[BUF];
	FILE *f, *p;

	ret = sscanf(command, "scan %d %d", &xend, &yend);

	if(ret < 2 || xend <= x || yend <= y
	   || xend > XPOS_MAX || yend > YPOS_MAX
	   || (xend - x)%PLOT_XINT != 0 || (yend - y)%PLOT_YINT != 0) {

		snprintf(buffer, BUF, "INVALID_VALUE\n");
		if(send(sock, buffer, strlen(buffer)+1, 0) < 0)
			return -1;
		
		return 0;
	}


	snprintf(buffer, BUF, "OK\n");
	if(send(sock, buffer, strlen(buffer)+1, 0) < 0)
		return -1;


	umask(0022);
	if((f = fopen(PLOT_DATFILE, "w")) == NULL)
		return -5;
	

	/* main loop; move around and take the analog value */
	for(i = y; i <= yend; i += PLOT_YINT) {
		for(j = x; j <= xend; j += PLOT_XINT) {
			if((i-y) % (2*PLOT_YINT) == 0)
				k = j;
			else
				k = x + xend - j;


			if((ret = recv(sock, buffer, BUF, MSG_DONTWAIT)) < 0 && errno != EAGAIN)
				return -1;
			if(ret >= 0) {
				if(strcmp(buffer, "cancel\n") == 0) {
					fclose(f);
					snprintf(buffer, BUF, "POS %d %d\n", k, i);
					if(send(sock, buffer, strlen(buffer)+1, 0) < 0)
						return -1;
		
					snprintf(buffer, BUF, "READY\n");
					if(send(sock, buffer, strlen(buffer)+1, 0) < 0)
						return -1;
		
					return 0;
				}
				else
					return -2;
			}
			snprintf(buffer, BUF, "POS %d %d\n", k, i);
			if(send(sock, buffer, strlen(buffer)+1, MSG_DONTWAIT) < 0 && 
			   errno != EAGAIN)
				return -1;


			if((ret = r8c_sr(r, w, "analog\n", buffer, BUF)) < 0)
				return ret;
			buffer[strlen(buffer) - 1] = '\0';	// remove '\n'
			value = atoi(buffer);

			fprintf(f, "%04d %04d %.4f\n", k, i, (float) value/1024*100);
#if DEBUG > 1
			printf("[cxa]: c_scan(): Writing values (x y z): "
			       "%04d %04d %.4f\n", k, i, (float) value/1024*100);
#endif

			/* don't move if we're already at the end point */
			if(j == xend)
				break;

			if((i-y) % (2*PLOT_YINT) == 0)
				snprintf(buffer, BUF, "stepper 1 rew %d\n", PLOT_XINT);
			else
				snprintf(buffer, BUF, "stepper 1 fwd %d\n", PLOT_XINT);
			if((ret = r8c_sr(r, w, buffer, NULL, 0)) < 0)
				return ret;
		}

		/* don't move if we're already at the end point */
		if(i == yend)
			break;

		fprintf(f, "\n");
#if DEBUG > 1
		printf("[cxa]: c_scan(): Writing values (x y z): \n");
#endif

		snprintf(buffer, BUF, "stepper 2 rew %d\n", PLOT_YINT);
		if((ret = r8c_sr(r, w, buffer, NULL, 0)) < 0)
			return ret;
	}
	fclose(f);


	/* genterate the picture (graph) */
	if((p = popen("gnuplot", "w")) == NULL)
		return -6;
	fputs("set terminal png\n", p);
	snprintf(buffer, BUF, "set output \'%s\'\n", PLOT_PICFILE);
	fputs(buffer, p);
	fputs("set grid\n", p);
	fputs("set dgrid3d 20,20,3\n", p);
	fputs("set contour base\n", p);
	fputs("set cntrparam levels 8\n", p);
	fputs("set xlabel \'x [cycles]\'\n", p);
	fputs("set ylabel \'y [cycles]\'\n", p);
	fputs("set zlabel \'potential [%]\'\n", p);
	fputs("set title \'Distribution of potential\'\n", p);
	snprintf(buffer, BUF, "splot \'%s\' with lines notitle\n", PLOT_DATFILE);
	fputs(buffer, p);
	pclose(p);
	umask(0077);


	snprintf(buffer, BUF, "POS %d %d\n", k, i);
	if(send(sock, buffer, strlen(buffer)+1, 0) < 0)
		return -1;
	snprintf(buffer, BUF, "READY\n");
	if(send(sock, buffer, strlen(buffer)+1, 0) < 0)
		return -1;

	return 0;
}

int c_line(int x, int y, int sock, FILE *r, FILE *w, const char *command) {
	int ret, value;
	int i;
	int end;
	unsigned char c;
	char buffer[BUF];
	FILE *f, *p;

	ret = sscanf(command, "line %c %d", &c, &end);

	if(ret < 2 || (c != 'x' && c != 'y')) {
		snprintf(buffer, BUF, "INVALID_VALUE\n");
		if(send(sock, buffer, strlen(buffer)+1, 0) < 0)
			return -1;
		
		return 0;
	}

	if(end <= (c == 'x' ? x : y) || end > (c == 'x' ? XPOS_MAX : YPOS_MAX)
	   || (c == 'x' ? ((end - x)%PLOT_XINT != 0) : ((end - y)%PLOT_YINT != 0))) {

		snprintf(buffer, BUF, "INVALID_VALUE\n");
		if(send(sock, buffer, strlen(buffer)+1, 0) < 0)
			return -1;
		
		return 0;
	}


	snprintf(buffer, BUF, "OK\n");
	if(send(sock, buffer, strlen(buffer)+1, 0) < 0)
		return -1;


	umask(0022);
	if((f = fopen(PLOT_DATFILE, "w")) == NULL)
		return -5;


	for(i = (c == 'x' ? x : y); i <= end; i += (c == 'x' ? PLOT_XINT : PLOT_YINT)) {
		if((ret = recv(sock, buffer, BUF, MSG_DONTWAIT)) < 0 && errno != EAGAIN)
			return -1;
		if(ret >= 0) {
			if(strcmp(buffer, "cancel\n") == 0) {
				fclose(f);
				snprintf(buffer, BUF, "POS %d %d\n", (c == 'x' ? i : x),
					 (c == 'y' ? i : y));
				if(send(sock, buffer, strlen(buffer)+1, 0) < 0)
					return -1;
		
				snprintf(buffer, BUF, "READY\n");
				if(send(sock, buffer, strlen(buffer)+1, 0) < 0)
					return -1;
		
				return 0;
			}
			else
				return -2;
		}
		snprintf(buffer, BUF, "POS %d %d\n", (c == 'x' ? i : x), (c == 'y' ? i : y));
		if(send(sock, buffer, strlen(buffer)+1, MSG_DONTWAIT) < 0 && 
		   errno != EAGAIN)
			return -1;


		if((ret = r8c_sr(r, w, "analog\n", buffer, BUF)) < 0)
			return ret;
		buffer[strlen(buffer) - 1] = '\0';	// remove '\n'
		value = atoi(buffer);

		fprintf(f, "%04d %.4f\n", i, (float) value/1024*100);
#if DEBUG > 1
		printf("[cxa]: c_line(): Writing values (x y): "
		       "%04d %.4f\n", i, (float) value/1024*100);
#endif

		/* don't move if we're already at the end point */
		if(i == end)
			break;

		snprintf(buffer, BUF, "stepper %d rew %d\n", (c == 'x' ? 1 : 2), 
			 (c == 'x' ? PLOT_XINT : PLOT_YINT));
		if((ret = r8c_sr(r, w, buffer, NULL, 0)) < 0)
			return ret;
	}
	fclose(f);


	/* genterate the picture (graph) */
	if((p = popen("gnuplot", "w")) == NULL)
		return -6;
	fputs("set terminal png\n", p);
	snprintf(buffer, BUF, "set output \'%s\'\n", PLOT_PICFILE);
	fputs(buffer, p);
	fputs("set grid\n", p);
	snprintf(buffer, BUF, "set xlabel \'%c [cycles]\'\n", c);
	fputs(buffer, p);
	fputs("set ylabel \'potential [%]\'\n", p);
	fputs("set title \'Distribution of potential\'\n", p);
	snprintf(buffer, BUF, "plot \'%s\' with linespoints notitle\n", PLOT_DATFILE);
	fputs(buffer, p);
	pclose(p);
	umask(0077);


	snprintf(buffer, BUF, "POS %d %d\n", (c == 'x' ? i : x), (c == 'y' ? i : y));
	if(send(sock, buffer, strlen(buffer)+1, 0) < 0)
		return -1;
	snprintf(buffer, BUF, "READY\n");
	if(send(sock, buffer, strlen(buffer)+1, 0) < 0)
		return -1;

	return 0;

}

