/* handlers.c -- sighandlers, errorhandler, daemonhandler
 * Copyright (C) 2008  theDark
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <termios.h>
#include <unistd.h>

#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>

#include "definitions.h"
#include "handlers.h"


void die_at_once() {
	fprintf(stderr, "[%d]: FATAL: Explicit kill of all children "
	        "was requested!\n", getpid());
	kill(0, SIGTERM);
	exit(EXIT_FAILURE);
}

void on_child_exit() {
	if(waitpid(-1, NULL, 0) < 0) {
		perror("waitpid()");
		die_at_once();
	}
}

void main_sigterm() {
	printf("[main]: Received SIGTERM, quitting...\n");
	kill(0, SIGTERM);
	exit(EXIT_FAILURE);
}

void main_sigint() {
	printf("[main]: Received SIGINT, quitting...\n");
	kill(0, SIGTERM);
	exit(EXIT_FAILURE);
}

void daemonize() {
	int i;
	pid_t pid;

	if((pid = fork()) < 0) {
		perror("[main]: fork()");
		die_at_once();
	}
	else if(pid != 0)
		exit(EXIT_SUCCESS);	// kill father
	
	if(setsid() < 0) {
		perror("[main]: setsid()");
		die_at_once();
	}

	if((pid = fork()) < 0) {
		perror("[main]: fork()");
		die_at_once();
	}
	else if(pid != 0)
		exit(EXIT_SUCCESS);	// kill father
	
	umask(0077);

	if(freopen(ERRORFILE, "a", stderr) == NULL) {
		perror("[main]: Could not open ERRORFILE");
		die_at_once();
	}
	if(freopen(LOGFILE, "a", stdout) == NULL) {
		perror("[main]: Could not open LOGFILE");
		die_at_once();
	}
	setvbuf(stderr, NULL, _IONBF, 0);
	setvbuf(stdout, NULL, _IOLBF, 0);
	
	for(i = sysconf(_SC_OPEN_MAX); i >= 0; i--) {
		if((i != STDERR_FILENO) && (i != STDOUT_FILENO))
			close(i);	// close all not needed fds
	}
}

int set_serial(FILE *r, FILE *w) {
	struct termios term;

	if(tcgetattr(fileno(r), &term) < 0)
		return -1;
	cfmakeraw(&term);
	term.c_iflag |= (IXON | IXOFF | IXANY);
	if(cfsetispeed(&term, BAUDRATE) < 0)
		return -1;
	if(cfsetospeed(&term, BAUDRATE) < 0)
		return -1;
	if(tcsetattr(fileno(r), TCSANOW, &term) < 0)
		return -1;

	if(tcgetattr(fileno(w), &term) < 0)
		return -1;
	cfmakeraw(&term);
	term.c_iflag |= (IXON | IXOFF | IXANY);
	if(cfsetispeed(&term, BAUDRATE) < 0)
		return -1;
	if(cfsetospeed(&term, BAUDRATE) < 0)
		return -1;
	if(tcsetattr(fileno(w), TCSANOW, &term) < 0)
		return -1;

	return 0;
}

