/* rcld -- Daemon to safely control an RCL (Remote Controlled Laboratory)
 * Copyright (C) 2008  theDark
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <time.h>
#include <fcntl.h>
#include <errno.h>
#include <unistd.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include "definitions.h"
#include "handlers.h"
#include "commands.h"


int main() {
	int _busy = 0;
	int _xpos = 0, _ypos = 0;
	time_t _time = 0;
	char _key[64] = "";

	pid_t pid;
	struct sigaction sig;
	struct sockaddr_un un;
	struct sockaddr_in in;
	int sock1, sock2, sock3;
	FILE *r, *w;
	int ret;
	unsigned int len;
	char buffer[BUF];
	char buf1[BUF], buf2[BUF];


#if DAEMON
	daemonize();
#endif


	printf("[main]: Starting with pid %d...\n", getpid());


	/* Let's create the child which handles complex actions (CXA-Child) */
	if((pid = fork()) < 0) {
		perror("[main]: fork()");
		die_at_once();
	}
	else if(pid == 0) {
		// CXA-Child

#if DEBUG
		printf("[cxa]: Starting with pid %d...\n", getpid());
#endif

		if((sock1 = socket(PF_UNIX, SOCK_DGRAM, 0)) < 0) {
			perror("[cxa]: Could not create unix-socket");
			die_at_once();
		}

		/* bind sock1 to recv() from main */
		memset(&un, 0, sizeof(un));
		un.sun_family = AF_UNIX;
		memcpy(un.sun_path, SUN_PATH3, SUN_PATH3_LENGTH);
		if(bind(sock1, (struct sockaddr *)&un, sizeof(un)) < 0) {
			perror("[cxa]: Could not bind unix-socket");
			die_at_once();
		}

		sleep(1);	// wait until main is ready
		
		/* connect sock1 to send() to main */
		memset(&un, 0, sizeof(un));
		un.sun_family = AF_UNIX;
		memcpy(un.sun_path, SUN_PATH1, SUN_PATH1_LENGTH);
		if(connect(sock1, (struct sockaddr *)&un, sizeof(un)) < 0) {
			perror("[cxa]: Could not connect to main");
			die_at_once();
		}

		if((r = fopen(SERIAL, "r")) == NULL) {
			perror("[cxa]: Could not open SERIAL for reading");
			die_at_once();
		}
		if((w = fopen(SERIAL, "w")) == NULL) {
			perror("[cxa]: Could not open SERIAL for writing");
			die_at_once();
		}
		if(set_serial(r, w) < 0) {
			fprintf(stderr, "[cxa]: Could not set up SERIAL connection\n");
			die_at_once();
		}
		setvbuf(r, NULL, _IOLBF, 0);
		setvbuf(w, NULL, _IOLBF, 0);


		for(;;) {
			if(recv(sock1, buf1, BUF, 0) < 0) {
				perror("[cxa]: Could not recv() from main");
				die_at_once();
			}

			/* detect command and do something */
			if(strncmp(buf1, "pos ", 4) == 0) {
				if(sscanf(buf1, "pos %d %d\n", &_xpos, &_ypos) < 2) {
					fprintf(stderr, "[cxa]: Error on getting pos from main\n");
					die_at_once();
				}
#if DEBUG > 1
				printf("[cxa]: Set pos to %d %d (sent by main)\n",
				       _xpos, _ypos);
#endif
			}
			else if(strcmp(buf1, "cancel\n") == 0) {
				// if we receive cancel at this point, we got just ready
				// while main was sending, so simply ignore this
				continue;
			}

			/* complex actions with r8c */
			else if(strcmp(buf1, "start") == 0) {
				if((ret = c_start(_xpos, _ypos, sock1, r, w)) < 0)
					make_c_error("c_start", ret);
			}
			else if(*buf1 == 'x' || *buf1 == 'y') {
				if((ret = c_xy(_xpos, _ypos, sock1, r, w, buf1)) < 0)
					make_c_error("c_xy", ret);
			}
#if USE_PLOT_FUNCTIONS
			else if(strncmp(buf1, "scan ", 5) == 0) {
				if((ret = c_scan(_xpos, _ypos, sock1, r, w, buf1)) < 0)
					make_c_error("c_scan", ret);
			}
			else if(strncmp(buf1, "line ", 5) == 0) {
				if((ret = c_line(_xpos, _ypos, sock1, r, w, buf1)) < 0)
					make_c_error("c_line", ret);
			}
#endif

			else {
				fprintf(stderr, "[cxa]: main did not send valid command for "
						"cxa, received stuff follows...\n");
				fprintf(stderr, "%s", buf1);
				die_at_once();
			}
		}

		fclose(w);
		fclose(r);
		close(sock1);
		exit(EXIT_SUCCESS);

	}


	/* Let's create the child which handles remote connections (CON-Child) */
	if((pid = fork()) < 0) {
		perror("[main]: fork()");
		die_at_once();
	}
	else if(pid == 0) {
		// CON-Child

#if DEBUG
		printf("[con]: Starting with pid %d...\n", getpid());
#endif

		sig.sa_handler = on_child_exit;
		sigemptyset(&sig.sa_mask);
		sig.sa_flags = SA_RESTART;
		sigaction(SIGCHLD, &sig, NULL);

		if((sock1 = socket(PF_INET, SOCK_STREAM, 0)) < 0) {
			perror("[con]: Could not create inet-socket");
			die_at_once();
		}
		memset(&in, 0, sizeof(in));
		in.sin_family = AF_INET;
		in.sin_port = htons(PORT);
		in.sin_addr.s_addr = htonl(INADDR_ANY);
		if(bind(sock1, (struct sockaddr *)&in, sizeof(in)) < 0) {
			perror("[con]: Could not bind inet-socket");
			die_at_once();
		}
		if(listen(sock1, MAX_CLIENTS) < 0) {
			perror("[con]: listen()");
			die_at_once();
		}

		sleep(1);	// wait until main is ready

		for(;;) {
			len = sizeof(in);
			if((sock2 = accept(sock1, (struct sockaddr *)&in, &len)) < 0) {
				perror("[con]: Could not accept client");
				continue;
			}

			if((pid = fork()) < 0) {
				perror("[con]: fork()");
				die_at_once();
			}
			else if(pid == 0) {
				// connection handler

				sig.sa_handler = SIG_IGN;
				sigemptyset(&sig.sa_mask);
				sig.sa_flags = SA_RESTART;
				sigaction(SIGPIPE, &sig, NULL);
				
#if DEBUG
				printf("[%d]: Client %s has connected\n",
				       getpid(), inet_ntoa(in.sin_addr));
#endif
				if((sock3 = socket(PF_UNIX, SOCK_DGRAM, 0)) < 0) {
					snprintf(buffer, BUF, "[%d]: Could not "
					         "create unix-socket", getpid());
					perror(buffer);
					exit(EXIT_FAILURE);
				}
				memset(&un, 0, sizeof(un));
				un.sun_family = AF_UNIX;
				snprintf(buffer, BUF, "%s%d", SUN_TMP_PATH, getpid());
				len = strlen(buffer);
				*buffer = '\0';
				memcpy(un.sun_path, buffer, len);
				if(bind(sock3, (struct sockaddr *)&un, sizeof(un)) < 0) {
					snprintf(buffer, BUF, "[%d]: Could not "
					         "bind unix-socket", getpid());
					perror(buffer);
					exit(EXIT_FAILURE);
				}
				memset(&un, 0, sizeof(un));
				un.sun_family = AF_UNIX;
				memcpy(un.sun_path, SUN_PATH2, SUN_PATH2_LENGTH);
				if(connect(sock3, (struct sockaddr *)&un, sizeof(un)) < 0) {
					snprintf(buffer, BUF, "[%d]: Could not "
					         "connect to main", getpid());
					perror(buffer);
					exit(EXIT_FAILURE);
				}
				
				while((ret = recv(sock2, buffer, BUF-1, 0)) > 0) {
					buffer[ret] = '\0';
					if(send(sock3, buffer, ret+1, 0) < 0) {
						snprintf(buffer, BUF, "[%d]: Could not "
						         "send() to main", getpid());
						perror(buffer);
						exit(EXIT_FAILURE);
					}
					if((ret = recv(sock3, buffer, BUF, 0)) < 0) {
						snprintf(buffer, BUF, "[%d]: Could not "
						         "recv() from main", getpid());
						perror(buffer);
						exit(EXIT_FAILURE);
					}
					if(send(sock2, buffer, ret, 0) < 0) {
						snprintf(buffer, BUF, "[%d]: Could not "
						         "send() to client", getpid());
						perror(buffer);
						exit(EXIT_FAILURE);
					}
				}
				if(ret < 0) {
					snprintf(buffer, BUF, "[%d]: Could not "
					         "recv() from client", getpid());
					perror(buffer);
					exit(EXIT_FAILURE);
				}
					
#if DEBUG
				printf("[%d]: Client %s has disconnected\n",
				       getpid(), inet_ntoa(in.sin_addr));
#endif

				close(sock2);
				close(sock3);
				exit(EXIT_SUCCESS);
			}

		}

		close(sock1);
		exit(EXIT_SUCCESS);
	}


	/* Let's go on with the parent */

	sig.sa_handler = on_child_exit;
	sigemptyset(&sig.sa_mask);
	sig.sa_flags = SA_RESTART;
	sigaction(SIGCHLD, &sig, NULL);

	sig.sa_handler = main_sigterm;
	sigemptyset(&sig.sa_mask);
	sig.sa_flags = SA_RESTART;
	sigaction(SIGTERM, &sig, NULL);

	sig.sa_handler = main_sigint;
	sigemptyset(&sig.sa_mask);
	sig.sa_flags = SA_RESTART;
	sigaction(SIGINT, &sig, NULL);

	if((sock1 = socket(PF_UNIX, SOCK_DGRAM, 0)) < 0) {
		perror("[main]: Could not create unix-socket");
		die_at_once();
	}
	if((sock2 = socket(PF_UNIX, SOCK_DGRAM, 0)) < 0) {
		perror("[main]: Could not create unix-socket");
		die_at_once();
	}

	/* bind sock2 to get commands from the clients */
	memset(&un, 0, sizeof(un));
	un.sun_family = AF_UNIX;
	memcpy(un.sun_path, SUN_PATH2, SUN_PATH2_LENGTH);
	if(bind(sock2, (struct sockaddr *)&un, sizeof(un)) < 0) {
		perror("[main]: Could not bind unix-socket");
		die_at_once();
	}
	
	/* bind sock1 to recv() from CXA-Child */
	memset(&un, 0, sizeof(un));
	un.sun_family = AF_UNIX;
	memcpy(un.sun_path, SUN_PATH1, SUN_PATH1_LENGTH);
	if(bind(sock1, (struct sockaddr *)&un, sizeof(un)) < 0) {
		perror("[main]: Could not bind unix-socket");
		die_at_once();
	}

	sleep(1);	// wait until cxa is ready
	
	/* connect sock1 to send() to CXA-Child */
	memset(&un, 0, sizeof(un));
	un.sun_family = AF_UNIX;
	memcpy(un.sun_path, SUN_PATH3, SUN_PATH3_LENGTH);
	if(connect(sock1, (struct sockaddr *)&un, sizeof(un)) < 0) {
		perror("[main]: Could not connect to cxa");
		die_at_once();
	}

	
	if((r = fopen(SERIAL, "r")) == NULL) {
		perror("[main]: Could not open SERIAL for reading");
		die_at_once();
	}
	if((w = fopen(SERIAL, "w")) == NULL) {
		perror("[main]: Could not open SERIAL for writing");
		die_at_once();
	}
	if(set_serial(r, w) < 0) {
		fprintf(stderr, "[main]: Could not set up SERIAL connection\n");
		die_at_once();
	}
	setvbuf(r, NULL, _IOLBF, 0);
	setvbuf(w, NULL, _IOLBF, 0);



	for(;;) {
		len = sizeof(un);
		if(recvfrom(sock2, buffer, BUF, 0, (struct sockaddr *)&un, &len) < 0) {
			perror("[main]: Could not recv() from some child");
			continue;
		}
#if DEBUG > 1
		printf("[main]: received: %s", buffer);
#endif
		if(splitstr(buffer, buf1, BUF, buf2, BUF) < 0) {
			snprintf(buffer, BUF, "ERROR: Bad formated string!\n");
			if(sendto(sock2, buffer, strlen(buffer), 0,
				  (struct sockaddr *)&un, len) < 0)
				perror("[main]: Could not send() to some child");

			continue;
		}


		/* if _busy was set, check whether r8c is still busy
		 * _busy == 1: CXA-Child is working with R8C
		 * _busy == 2: CXA-Child is working with R8C, but cancel was or cannot be sent
		 */
		if(_busy) {
			while((ret = recv(sock1, buffer, BUF, MSG_DONTWAIT)) > 0) {
				if(strcmp(buffer, "READY\n") == 0) {
					// CXA-Child has finished its work, we're no longer _busy
					_busy = 0;
				}
				else if(strncmp(buffer, "POS ", 4) == 0) {
					if(sscanf(buffer, "POS %d %d\n", &_xpos, &_ypos) < 2) {
						fprintf(stderr, "[main]: Error on getting POS "
							"from cxa\n");
						die_at_once();
					}
#if DEBUG > 1
					printf("[main]: Set pos to %d %d (sent by cxa)\n",
					       _xpos, _ypos);
#endif
				}
				else {
					fprintf(stderr, "[main]: CXA-Child did not send POS "
							"or READY, received stuff follows...\n");
					fprintf(stderr, "%s", buffer);
					die_at_once();
				}
			}
			if(ret < 0) {
				if(errno != EAGAIN) {
					perror("[main]: Error on reading from CXA-Child");
					die_at_once();
				}
#if DEBUG > 1
				else if(_busy)
					printf("[main]: CXA-Child is still busy\n");
#endif
			}
		}


		/* detect command and do something */
		if(strcmp(buf2, "getpos") == 0) {
			snprintf(buffer, BUF, "%d %d\n", _xpos, _ypos);
			if(sendto(sock2, buffer, strlen(buffer), 0,
				  (struct sockaddr *)&un, len) < 0)
				perror("[main]: Could not send() to some child");

			continue;
		}
		if(strcmp(buf2, "gettime") == 0) {
			snprintf(buffer, BUF, "%ld\n", USE_TIME - (time(NULL) - _time));
			if(sendto(sock2, buffer, strlen(buffer), 0,
				  (struct sockaddr *)&un, len) < 0)
				perror("[main]: Could not send() to some child");

			continue;
		}
		if(strcmp(buf2, "getusetime") == 0) {
			snprintf(buffer, BUF, "%d\n", USE_TIME);
			if(sendto(sock2, buffer, strlen(buffer), 0,
				  (struct sockaddr *)&un, len) < 0)
				perror("[main]: Could not send() to some child");

			continue;
		}
		if(strcmp(buf2, "getxposmax") == 0) {
			snprintf(buffer, BUF, "%d\n", XPOS_MAX);
			if(sendto(sock2, buffer, strlen(buffer), 0,
				  (struct sockaddr *)&un, len) < 0)
				perror("[main]: Could not send() to some child");

			continue;
		}
		if(strcmp(buf2, "getyposmax") == 0) {
			snprintf(buffer, BUF, "%d\n", YPOS_MAX);
			if(sendto(sock2, buffer, strlen(buffer), 0,
				  (struct sockaddr *)&un, len) < 0)
				perror("[main]: Could not send() to some child");

			continue;
		}
#if USE_PLOT_FUNCTIONS
		if(strcmp(buf2, "getplotxint") == 0) {
			snprintf(buffer, BUF, "%d\n", PLOT_XINT);
			if(sendto(sock2, buffer, strlen(buffer), 0,
				  (struct sockaddr *)&un, len) < 0)
				perror("[main]: Could not send() to some child");

			continue;
		}
		if(strcmp(buf2, "getplotyint") == 0) {
			snprintf(buffer, BUF, "%d\n", PLOT_YINT);
			if(sendto(sock2, buffer, strlen(buffer), 0,
				  (struct sockaddr *)&un, len) < 0)
				perror("[main]: Could not send() to some child");

			continue;
		}
		if(strcmp(buf2, "getplotdatfile") == 0) {
			snprintf(buffer, BUF, "%s\n", PLOT_DATFILE);
			if(sendto(sock2, buffer, strlen(buffer), 0,
				  (struct sockaddr *)&un, len) < 0)
				perror("[main]: Could not send() to some child");

			continue;
		}
		if(strcmp(buf2, "getplotpicfile") == 0) {
			snprintf(buffer, BUF, "%s\n", PLOT_PICFILE);
			if(sendto(sock2, buffer, strlen(buffer), 0,
				  (struct sockaddr *)&un, len) < 0)
				perror("[main]: Could not send() to some child");

			continue;
		}
#endif
		if(strcmp(buf2, "loggedin") == 0) {
			if(strcmp(_key, buf1) == 0)
				snprintf(buffer, BUF, "YES\n");
			else if(*_key == '\0')
				snprintf(buffer, BUF, "NO\n");
			else
				snprintf(buffer, BUF, "IN_USE\n");

			if(sendto(sock2, buffer, strlen(buffer), 0,
				  (struct sockaddr *)&un, len) < 0)
				perror("[main]: Could not send() to some child");

			continue;
		}

		if(strcmp(buf2, "login") == 0) {
			if(strcmp(_key, buf1) == 0 && time(NULL) - _time < USE_TIME) {
				snprintf(buffer, BUF, "ERROR: Already logged in!\n");
				if(sendto(sock2, buffer, strlen(buffer), 0,
					  (struct sockaddr *)&un, len) < 0)
					perror("[main]: Could not send() to some child");
			}
			else if(*_key != '\0' && time(NULL) - _time < USE_TIME) {
				snprintf(buffer, BUF, "IN_USE\n");
				if(sendto(sock2, buffer, strlen(buffer), 0,
					  (struct sockaddr *)&un, len) < 0)
					perror("[main]: Could not send() to some child");
			}
			else {
				strncpy(_key, buf1, 64);
				_key[63] = '\0';
				_time = time(NULL);
#if DEBUG
				printf("[main]: Client logged in (key: %s)\n", _key);
#endif
				snprintf(buffer, BUF, "OK\n");
				if(sendto(sock2, buffer, strlen(buffer), 0,
					  (struct sockaddr *)&un, len) < 0)
					perror("[main]: Could not send() to some child");
			}
			continue;
		}


		/* the following commands require a valid login, so check this */
		if(strcmp(_key, buf1) != 0) {
			snprintf(buffer, BUF, "NOT_LOGGED_IN\n");
			if(sendto(sock2, buffer, strlen(buffer), 0,
				  (struct sockaddr *)&un, len) < 0)
				perror("[main]: Could not send() to some child");

			continue;
		}

		/* commands that need a valid login */
		if(strcmp(buf2, "logout") == 0) {
#if DEBUG
			printf("[main]: Client logged out (key: %s)\n", _key);
#endif
			*_key = '\0';

			snprintf(buffer, BUF, "OK\n");
			if(sendto(sock2, buffer, strlen(buffer), 0,
				  (struct sockaddr *)&un, len) < 0)
				perror("[main]: Could not send() to some child");

			continue;
		}
		if(strcmp(buf2, "cancel") == 0) {
			if(_busy != 1)
				snprintf(buffer, BUF, "NOTHING_TO_CANCEL\n");
			else {
				snprintf(buffer, BUF, "cancel\n");
				if(send(sock1, buffer, strlen(buffer)+1, 0) < 0) {
					perror("[main]: Could not send() to CXA-Child");
					die_at_once();
				}
				_busy = 2;
				snprintf(buffer, BUF, "OK\n");
			}
			if(sendto(sock2, buffer, strlen(buffer), 0,
				  (struct sockaddr *)&un, len) < 0)
				perror("[main]: Could not send() to some child");

			continue;
		}


		/* the following commands require a not busy r8c, so check this */
		if(_busy) {
			snprintf(buffer, BUF, "BUSY\n");
			if(sendto(sock2, buffer, strlen(buffer), 0,
				  (struct sockaddr *)&un, len) < 0)
				perror("[main]: Could not send() to some child");

			continue;
		}

		/* commands that need a not busy r8c */
		if(strcmp(buf2, "analog") == 0) {
			if(r8c_sr(r, w, "analog\n", buffer, BUF) < 0) {
				fprintf(stderr, "[main]: Error on sending to or receiving from R8C\n");
				die_at_once();
			}

			buffer[strlen(buffer) - 1] = '\0';	// remove '\n'
			ret = atoi(buffer);
			snprintf(buffer, BUF, "%.2f%%\n", (float) ret/1024*100);

			if(sendto(sock2, buffer, strlen(buffer), 0,
				  (struct sockaddr *)&un, len) < 0)
				perror("[main]: Could not send() to some child");

			continue;
		}
		/* the following commands are handled by the CXA-Child */
		if(strcmp(buf2, "start") == 0
		   || *buf2 == 'x' || *buf2 == 'y'
#if USE_PLOT_FUNCTIONS
		   || strncmp(buf2, "scan ", 5) == 0
		   || strncmp(buf2, "line ", 5) == 0
#endif
		  ) {
			snprintf(buffer, BUF, "pos %d %d\n", _xpos, _ypos);
			if(send(sock1, buffer, strlen(buffer)+1, 0) < 0) {
				perror("[main]: Could not send() to CXA-Child");
				die_at_once();
			}
			if(send(sock1, buf2, strlen(buf2)+1, 0) < 0) {
				perror("[main]: Could not send() to CXA-Child");
				die_at_once();
			}
			if(recv(sock1, buffer, BUF, 0) < 0) {
				perror("[main]: Could not recv() from CXA-Child");
				die_at_once();
			}
			
			/* if answer from cxa is OK, cxa is busy; if not nothing will be done */
			if(strcmp(buffer, "OK\n") == 0) {
				if(*buf2 == 'x' || *buf2 == 'y')
					_busy = 2; // this command cannot be cancelled
				else
					_busy = 1;
			}

			if(sendto(sock2, buffer, strlen(buffer), 0,
				  (struct sockaddr *)&un, len) < 0)
				perror("[main]: Could not send() to some child");

			continue;
		}


		/* if we're still here, this is an unknown command */
		snprintf(buffer, BUF, "ERROR: Unknown command: %s\n", buf2);
		if(sendto(sock2, buffer, strlen(buffer), 0,
			  (struct sockaddr *)&un, len) < 0)
			perror("[main]: Could not send() to some child");

	}

	fclose(w);
	fclose(r);
	close(sock2);
	close(sock1);

	return EXIT_SUCCESS;
}

