/* commands.c -- Available commands on the commandline
 * Copyright (C) 2008  theDark
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "commands.h"
#include "generic.h"
#include "serial.h"
#include "stepper.h"
#include "display.h"
#include "sfr_r813.h"


void c_help(char *command) {
	
	if(eq(command, "led")) {
		writetxd0("Usage: led <id> <state>\n");
		writetxd0("<id>: [0-3]\n");
		writetxd0("<state>: on/off\n");
	}
	else if(eq(command, "lcd")) {
		writetxd0("Usage: lcd <command> [arg1] [arg2]\n");
		writetxd0("<command>: clear, shift, pos, write\n");
		writetxd0("clear: Restores everything and clears the display\n");
		writetxd0("shift <direction>: Shifts display; <direction> = left/right/restore\n");
		writetxd0("pos <line> [row]: Puts the cursor to line <line> and row [row]\n");
		writetxd0("                  line: [1-2]; row: [1-40]; row is set to 1 if not specified\n");
		writetxd0("write <string>: writes <string> at cursor position\n"
			  "                (<string> should be doublequoted)\n");
	}
	else if(eq(command, "analog")) {
		writetxd0("Usage: analog [ignored args]\n");
		writetxd0("Returns the voltage at input port p0_1/an6 as 10bit value!\n");
	}
	else if(eq(command, "stepper")) {
		writetxd0("Usage: stepper <id> <direction> [cycles]\n");
		writetxd0("<id>: [1-2]\n");
		writetxd0("<direction>: fwd/rew\n");
		writetxd0("[cycles]: [1-10000] (default: 1)\n");
		writetxd0("25 cycles are one turn!\n");
	}
	else if(eq(command, "key")) {
		writetxd0("Usage: key <state>\n");
		writetxd0("<state>: on/off\n");
		writetxd0("Switches the key input interrupt on/off!\n");
	}

	else if(*command == '\0') {
		writetxd0("Usage: <command> [arg1] [arg2] ... [argN]\n");
		writetxd0("For command-specific help try: help <command>\n");
	}
	else {
		writetxd0("There is no help for command: \"");
		writetxd0(command);
		writetxd0("\"\n");
	}
}

void c_led(char *id, char *state) {
	char byte;
	
	if(id[0] < '0' || id[0] > '3' || id[1] != '\0') {
		writetxd0("Unknown id: \"");
		writetxd0(id);
		writetxd0("\"\n");
		return;
	}

	byte = 0x01 << (*id - '0');
	pd1 |= byte;

	if(eq(state, "on")) {
		p1 |= byte;
		writetxd0("OK\n");
	}
	else if(eq(state, "off")) {
		p1 &= ~(byte);
		writetxd0("OK\n");
	}
	else {
		writetxd0("Unknown state: \"");
		writetxd0(state);
		writetxd0("\"\n");
	}
}

void c_lcd(char *command, char *string, char *row) {
	int r = 0;

	if(eq(command, "clear")) {
		lcdctrl(0x01);	// clear display
		writetxd0("OK\n");
	}
	else if(eq(command, "shift")) {
		if(eq(string, "left")) {
			lcdctrl(0x18);
			writetxd0("OK\n");
		}
		else if(eq(string, "right")) {
			lcdctrl(0x1c);
			writetxd0("OK\n");
		}
		else if(eq(string, "restore")) {
			lcdctrl(0x02);
			writetxd0("OK\n");
		}
		else {
			writetxd0("Unknown direction: \"");
			writetxd0(string);
			writetxd0("\"\n");
		}
	}
	else if(eq(command, "pos")) {
		if(string[0] < '1' || string[0] > '2' || string[1] != '\0') {
			writetxd0("Unknown line: \"");
			writetxd0(string);
			writetxd0("\"\n"); // in this case: string == line !
			return;
		}
		
		if(*row == '\0')
			r = 1;
		else if(row[1] == '\0' && row[0] >= '0' && row[0] <= '9')
			r = *row - '0';
		else if(row[2] == '\0' && row[0] >= '0' && row[0] <= '9'
				       && row[1] >= '0' && row[1] <= '9')
			r = (row[0] - '0')*10 + row[1] - '0';

		if(r < 1 || r > 40) {
			writetxd0("Unknown row: \"");
			writetxd0(row);
			writetxd0("\"\n");
			return;
		}

		lcdpos(*string - '0', r);
		writetxd0("OK\n");
	}
	else if(eq(command, "write")) {
		writelcd(string);
		writetxd0("OK\n");
	}
	else {
		writetxd0("Unknown lcd-command: \"");
		writetxd0(command);
		writetxd0("\"\n");
	}
}

void c_analog() {
	unsigned int value;
	unsigned char c;

	prc2 = 1;		// disable protection
	pd0_1 = 0;		// input port

	adcon1 = 0x28;		// 10bit mode; vref connected
	adcon0 = 0x00 + 6;	// one-shot mode; input an6
	adst = 1;		// start conversion

	while(adst)		// wait until end of conversion
		wdtr = 0x00;
	
	value = ad;		// get 10bit result

	c = value/1000;
	value -= c*1000;
	sendtxd0(c + '0');
	c = value/100;
	value -= c*100;		// send result as string
	sendtxd0(c + '0');
	c = value/10;
	value -= c*10;
	sendtxd0(c + '0');
	sendtxd0(value + '0');
	sendtxd0('\n');
}

void c_stepper(char *id, char *direction, char *cycles) {
	int i, len, fail = 0, mul = 1;
	unsigned int n = 0;

	if(*cycles != '\0') {
		for(len = 0; cycles[len] != '\0'; len++);
		
		for(i = len-1; i >= 0; i--, mul *= 10) {
			if(cycles[i] < '0' || cycles[i] > '9') {
				fail = 1;
				break;
			}
			n += (cycles[i] - '0') * mul;
		}
	}
	else
		n = 1;

	if(fail || n > 10000) {
		fail = 0;
		writetxd0("Unknown cycle-number: \"");
		writetxd0(cycles);
		writetxd0("\"\n");
		return;
	}

	if(eq(direction, "fwd")) {
		if(eq(id, "1")) {
			stepper_fwd(&pd1, &p1, n);
			writetxd0("OK\n");
		}
		else if(eq(id, "2")) {
			stepper_fwd(&pd3, &p3, n);
			writetxd0("OK\n");
		}
		else
			fail = 1;
	}
	else if(eq(direction, "rew")) {
		if(eq(id, "1")) {
			stepper_rew(&pd1, &p1, n);
			writetxd0("OK\n");
		}
		else if(eq(id, "2")) {
			stepper_rew(&pd3, &p3, n);
			writetxd0("OK\n");
		}
		else
			fail = 1;
	}
	else {
		writetxd0("Unknown direction: \"");
		writetxd0(direction);
		writetxd0("\"\n");
		return;
	}

	if(fail) {
		writetxd0("Unknown stepper-id: \"");
		writetxd0(id);
		writetxd0("\"\n");
	}
}

void c_key(char *state) {

	if(eq(state, "on")) {
		pd1_0 = 0;
		ki0en = 1;
		ki0pl = 1;
		kupic = 1;
		writetxd0("OK\n");
	}
	else if(eq(state, "off")) {
		kupic = 0;
		ki0en = 0;
		writetxd0("OK\n");
	}
	else {
		writetxd0("Unknown state: \"");
		writetxd0(state);
		writetxd0("\"\n");
	}
}

